import { ConfigState, PriceResult } from './types';
import { PRICES } from './constants';

export const calculatePrice = (state: ConfigState): PriceResult => {
  let s1 = 0, s2 = 0, s3 = 0, s4 = 0, s5 = 0, s6 = 0;
  const area = state.areaInput || 0;

  // 1. Planung & Fundament
  if (state.checkBauantrag) s1 += PRICES.bauantrag;
  if (state.checkKfW) s1 += area * PRICES.kfw;
  if (state.checkSoilReport) s1 += PRICES.soilReport;
  if (state.checkSurveying) s1 += PRICES.surveying;
  if (state.checkCrane) s1 += PRICES.crane;
  
  if (state.fundamentSelect !== '0') {
      s1 += state.fundamentArea * PRICES.fund[state.fundamentSelect];
  }

  // 2. Haus & Konstruktion
  // Außenwände (Basis)
  s2 += area * PRICES.wallExterior[state.wallThicknessExterior];
  
  // Innenwände (Zu/Abschlag)
  s2 += area * PRICES.innerWallSurcharge[state.wallThicknessInterior];

  // Wandhöhe Aufschlag
  if (state.wallHeight > 245) {
      s2 += (state.wallHeight - 245) * PRICES.height * area;
  }

  if (state.roofSelect !== 'none') {
      s2 += area * PRICES.roof[state.roofSelect];
  }

  // Fassade Logic
  const wallArea = Math.sqrt(area) * 4 * (state.wallHeight / 100);
  
  if (state.facadeSelect !== 'paneele' && state.facadeSelect !== 'none') {
      s2 += wallArea * PRICES.facade[state.facadeSelect];
  }

  // --- COMPLEX ENERGY LOGIC (REFINED) ---
  let score = 0;
  let uValue = "0.24"; // Base fallback
  let energyStandard = "GEG 2024";

  // 1. Hülle (Außenwand) - Basis der Isolierung (Max 35 Pkt)
  if (state.wallThicknessExterior === '15') { 
      score += 15; 
      uValue = "0.20"; 
  } else if (state.wallThicknessExterior === '17') { 
      score += 28; 
      uValue = "0.16"; 
      energyStandard = "KfW 40 Ready"; 
  } else if (state.wallThicknessExterior === '20') { 
      score += 35; 
      uValue = "0.13"; 
      energyStandard = "Passivhaus"; 
  }

  // 2. Speichermasse (Innenwand) - (Max 20 Pkt)
  // Deutlich erhöhter Einfluss, da thermische Masse Schwankungen ausgleicht
  if (state.wallThicknessInterior === '12') {
      score += 0;   // Kaum Masse, schnelle Überhitzung/Auskühlung
  } else if (state.wallThicknessInterior === '15') {
      score += 10;  // Guter Standard
  } else if (state.wallThicknessInterior === '20') {
      score += 20;  // Exzellente Speichermasse (Premium)
  }

  // 3. Dachdämmung (Max 5 Pkt)
  // Dach ist wichtig, aber weniger Varianz im Tool als Wände
  if (state.roofSelect !== 'none') score += 5;

  // 4. Fenster (Max 5 Pkt)
  if (state.windowMaterial === 'alu') score += 5; // Thermisch getrenntes Alu vs Standard Kst

  // 5. Technik & Heizung (Max 25 Pkt)
  if (state.heatSelect === 'floor_elec') score += 5; // Ineffizient bzgl Primärenergie
  else if (state.heatSelect === 'ac') score += 12; // Luft-Luft WP (gut, aber wenig Trägheit)
  else if (state.heatSelect === 'floor_water') score += 18; // Wassergeführt (gut mit Masse)
  else if (state.heatSelect === 'pump') score += 25; // Luft-Wasser WP (Top Standard)

  // 6. Lüftung (Max 10 Pkt)
  // Essentiell für Dichtheit und Wärmerückgewinnung
  if (state.ventSelect !== '0') score += 10;

  // Cap at 100
  // Zusatzpunkte für PV System, um auf 100 zu kommen, wenn Hülle nicht perfekt ist, 
  // oder um den Score zu "boosten"
  if (state.pvSystem !== 'none') score += 5;

  const energyScore = Math.min(score, 100);

  // 3. Fenster & Türen
  const winFactor = PRICES.winMaterialFactor[state.windowMaterial];

  s3 += ((state.winPano || 0) * PRICES.win.pano);
  s3 += ((state.winStd || 0) * PRICES.win.std) * winFactor;
  s3 += (state.shutterCount || 0) * PRICES.win.shutter;
  
  s3 += (state.doorOut || 0) * PRICES.door.out;
  s3 += (state.doorIn || 0) * PRICES.door.in;
  
  const glassPrice = state.glassDoorFrame === 'plastic' ? PRICES.door.glassKst : PRICES.door.glassAlu;
  s3 += (state.glassDoorIn || 0) * glassPrice;

  // 4. Innenausbau
  let interiorCalculationArea = area * 2.5; 
  if (state.checkCeilingIncluded) {
      interiorCalculationArea += area;
  }

  if (state.drywallSelect !== 'none') {
      s4 += interiorCalculationArea * PRICES.dry[state.drywallSelect];
  }

  if (state.paintSelect !== 'none') {
      let pPrice = PRICES.paint.std;
      if (state.paintSelect === 'color') pPrice *= PRICES.paint.factor;
      s4 += interiorCalculationArea * pPrice;
  }

  if (state.floorSelect !== 'none') {
      s4 += area * PRICES.floor[state.floorSelect];
      if (state.floorSelect === 'laminat' && state.checkTrittschall) {
          s4 += area * PRICES.trittschall;
      }
  }

  s4 += (state.socketCount || 0) * PRICES.elec.socket;
  s4 += (state.lanCount || 0) * PRICES.elec.lan;
  s4 += (state.lightStdCount || 0) * PRICES.elec.lightStd;
  s4 += (state.lightSpotCount || 0) * PRICES.elec.lightSpot;
  s4 += (state.smokeCount || 0) * PRICES.elec.smoke;
  if (state.checkFuseBox) s4 += PRICES.elec.fuse;

  // 5. Technik & Sanitär
  if (state.heatSelect === 'ac') {
      s5 += (state.acCount || 0) * PRICES.ac;
  }
  if (state.heatSelect === 'floor_elec') s5 += area * PRICES.heatFloor.el;
  if (state.heatSelect === 'floor_water') s5 += area * PRICES.heatFloor.wa;
  if (state.heatSelect === 'pump') {
      const load = area * 0.05;
      s5 += load * PRICES.heatPump;
  }
  if (state.checkPvPrep) s5 += PRICES.pvPrep;
  
  if (state.pvSystem !== 'none') {
      s5 += PRICES.pvSystem[state.pvSystem];
  }
  
  if (state.batteryStorage !== 'none') {
      s5 += PRICES.batteryStorage[state.batteryStorage];
  }

  if (state.waterType !== 'none' && state.waterModel) {
      if (state.waterType === 'boiler') {
           const model = state.waterModel as keyof typeof PRICES.water.boiler;
           if (PRICES.water.boiler[model]) s5 += PRICES.water.boiler[model];
      } else if (state.waterType === 'flow') {
           const model = state.waterModel as keyof typeof PRICES.water.flow;
           if (PRICES.water.flow[model]) s5 += PRICES.water.flow[model];
      }
  }

  s5 += (state.wcCount || 0) * PRICES.sanitary.wc;
  s5 += (state.sinkCount || 0) * PRICES.sanitary.sink;
  s5 += (state.showerCount || 0) * PRICES.sanitary.shower;
  s5 += (state.tubCount || 0) * PRICES.sanitary.tub;
  s5 += (state.tapCount || 0) * PRICES.sanitary.tap;

  // 6. Extras
  s6 += (state.terraceArea || 0) * PRICES.terrace;
  s6 += (state.garageCount || 0) * PRICES.garage;
  if (state.checkStairOut) s6 += PRICES.stair.out;
  if (state.checkStairIn) s6 += PRICES.stair.in;

  const netto = s1 + s2 + s3 + s4 + s5 + s6;
  const tax = netto * 0.19;
  const brutto = netto + tax;

  return {
      s1, s2, s3, s4, s5, s6,
      netto, tax, brutto,
      wallArea, uValue, energyStandard, energyScore
  };
};

export const formatCurrency = (val: number) => {
  return new Intl.NumberFormat('de-DE', { style: 'currency', currency: 'EUR' }).format(val);
};