import React from 'react';
import { ConfigState, PriceResult } from '../types';
import { formatCurrency } from '../utils';
import RoofPreview from './RoofPreview';

interface SummarySidebarProps {
  config: ConfigState;
  result: PriceResult;
  onOpenPDF: () => void;
}

const SummarySidebar: React.FC<SummarySidebarProps> = ({ config, result, onOpenPDF }) => {
  return (
    <div className="hidden lg:block sticky top-20 h-fit">
       <div className="bg-white rounded-xl shadow-lg border border-gray-200 overflow-hidden transition-all duration-500 hover:shadow-xl">
          {/* Visual Preview */}
          <div className="p-8 text-center bg-gray-50 border-b border-gray-200 relative group">
            <div className="transition-transform duration-500 group-hover:scale-105">
                <RoofPreview roof={config.roofSelect} facade={config.facadeSelect} />
            </div>
            <div className="mt-4 font-serif text-xl text-primary tracking-wider">
               {config.roofSelect === 'sattel' ? 'Satteldach' : config.roofSelect === 'pult' ? 'Pultdach' : config.roofSelect === 'flat' ? 'Flachdach' : 'Modul Mate'}
            </div>
            <div className="text-xs text-gray-400 mt-1 uppercase tracking-widest">{config.areaInput} m² Wohnfläche</div>
          </div>
          
          {/* Price Block */}
          <div className="bg-primary text-white p-8 text-center border-b-4 border-accent relative overflow-hidden">
             <div className="absolute top-0 left-0 w-full h-full bg-[url('https://www.transparenttextures.com/patterns/cubes.png')] opacity-10"></div>
             <div className="relative z-10">
                 <div className="text-xs tracking-[0.2em] uppercase text-gray-400 mb-2">Gesamtpreis (Netto)</div>
                 <div className="text-4xl lg:text-5xl my-3 text-accent leading-none font-serif">{formatCurrency(result.netto)}</div>
                 <div className="text-sm opacity-80 font-light">zzgl. 19% MwSt: {formatCurrency(result.tax)}</div>
                 <div className="mt-4 pt-4 border-t border-white/10 flex justify-between items-center text-sm font-medium">
                    <span>Brutto</span>
                    <span className="text-lg">{formatCurrency(result.brutto)}</span>
                 </div>
             </div>
          </div>

          {/* Features List */}
          <div className="p-8 bg-white">
             <h4 className="m-0 mb-4 text-xs font-bold tracking-widest text-primary uppercase border-b border-gray-100 pb-2">Inklusivleistungen</h4>
             <ul className="space-y-3 text-sm text-gray-600">
                {['Lieferung (LKW)', 'Montage & Aufbau', 'Hausanschlüsse', 'Architektenleistung (Basis)'].map((item, i) => (
                  <li key={i} className="flex items-center group relative">
                    <span className="w-5 h-5 rounded-full bg-green-50 flex items-center justify-center mr-3 group-hover:bg-green-100 transition-colors">
                        <i className="fas fa-check text-xs text-success"></i>
                    </span>
                    {item}
                    {item === 'Hausanschlüsse' && (
                        <div className="ml-1 group/info relative inline-block cursor-help">
                            <i className="fas fa-info-circle text-gray-300 hover:text-accent text-xs transition-colors"></i>
                            <div className="invisible group-hover/info:visible absolute z-50 bottom-[140%] left-1/2 -ml-[100px] w-48 bg-primary text-white text-left text-xs p-3 rounded shadow-lg opacity-0 group-hover/info:opacity-100 transition-opacity pointer-events-none after:content-[''] after:absolute after:top-full after:left-1/2 after:-ml-[5px] after:border-[5px] after:border-solid after:border-t-primary after:border-x-transparent after:border-b-transparent leading-relaxed">
                                Anschlüsse für Wasser, Strom und Abwasser bis zur Gebäudeaußenkante (Medienerschließung).
                            </div>
                        </div>
                    )}
                  </li>
                ))}
             </ul>
          </div>

          {/* Action */}
          <div className="p-6 bg-gray-50 border-t border-gray-200">
             <button 
               onClick={onOpenPDF}
               className="group w-full bg-primary text-accent border border-primary py-4 px-6 rounded-lg text-lg font-semibold cursor-pointer uppercase tracking-widest transition-all flex items-center justify-center gap-3 shadow-md hover:bg-accent hover:text-white hover:border-accent hover:shadow-lg active:scale-95"
             >
                <i className="fas fa-file-pdf transition-transform group-hover:-translate-y-1"></i> 
                <span>Angebot PDF</span>
             </button>
             <p className="text-center text-[10px] text-gray-400 mt-4 leading-normal">
               Unverbindliche Preiskalkulation.<br/>Preise zzgl. Baunebenkosten.
             </p>
          </div>
       </div>
    </div>
  );
};

export default SummarySidebar;