import React from 'react';

interface CheckboxCardProps {
  checked: boolean;
  onChange: (checked: boolean) => void;
  title: string;
  subtitle?: string | React.ReactNode;
  tooltip?: string;
  compact?: boolean;
  onChat?: () => void; // New prop for Chatbot trigger
}

const CheckboxCard: React.FC<CheckboxCardProps> = ({ checked, onChange, title, subtitle, tooltip, compact, onChat }) => {
  return (
    <label 
      className={`
        flex gap-4 border border-gray-200 rounded-md cursor-pointer transition-all duration-200 bg-white hover:border-accent hover:bg-gray-50 relative group/card
        ${checked ? '!border-accent !bg-accent-light' : ''}
        ${compact ? 'p-3 items-center h-auto' : 'p-5 items-start h-full'}
      `}
    >
      <input 
        type="checkbox" 
        checked={checked} 
        onChange={(e) => onChange(e.target.checked)}
        className="mt-1 accent-accent scale-125"
        style={{ marginTop: compact ? 0 : '4px' }}
      />
      <div className="flex-1">
        <div className="flex items-center gap-2 flex-wrap">
            <strong className={`block text-primary text-base ${compact ? 'mb-0 text-[0.95rem]' : 'mb-1'} ${checked ? 'text-accent' : ''}`}>
            {title}
            </strong>
            {/* AI Help Button */}
            {onChat && (
              <div 
                onClick={(e) => { e.preventDefault(); e.stopPropagation(); onChat(); }}
                className="ml-auto md:ml-2 w-6 h-6 rounded-full bg-gradient-to-tr from-accent to-[#f1dca7] text-white flex items-center justify-center hover:scale-110 transition-transform shadow-sm cursor-pointer z-10"
                title="KI-Berater fragen"
              >
                 <i className="fas fa-sparkles text-[10px]"></i>
              </div>
            )}

            {compact && subtitle && <span className="text-sm text-gray-500 w-full md:w-auto">{subtitle}</span>}
        </div>
        
        {!compact && subtitle && <small className="text-gray-500 text-sm leading-snug block">{subtitle}</small>}
      </div>
      
      {tooltip && (
        <div className="group relative hidden md:block">
          <i className="fas fa-question-circle text-gray-300 hover:text-accent cursor-help text-sm transition-colors"></i>
          <div className="invisible group-hover:visible absolute z-50 bottom-[140%] left-1/2 -ml-[120px] w-60 bg-primary text-white text-left text-xs p-3 rounded shadow-lg opacity-0 group-hover:opacity-100 transition-opacity pointer-events-none after:content-[''] after:absolute after:top-full after:left-1/2 after:-ml-[5px] after:border-[5px] after:border-solid after:border-t-primary after:border-x-transparent after:border-b-transparent">
            {tooltip}
          </div>
        </div>
      )}
    </label>
  );
};

export default CheckboxCard;