import React, { useState, useEffect, useRef } from 'react';
import { sendMessageToGemini } from '../ai';

interface Message {
  role: 'user' | 'model';
  text: string;
}

interface ChatWidgetProps {
  isOpen: boolean;
  onClose: () => void;
  context: string; // The topic the user clicked on
}

const ChatWidget: React.FC<ChatWidgetProps> = ({ isOpen, onClose, context }) => {
  const [messages, setMessages] = useState<Message[]>([]);
  const [input, setInput] = useState('');
  const [loading, setLoading] = useState(false);
  const scrollRef = useRef<HTMLDivElement>(null);
  const [hasInitialized, setHasInitialized] = useState(false);

  // Auto-scroll to bottom
  useEffect(() => {
    if (scrollRef.current) {
      scrollRef.current.scrollTop = scrollRef.current.scrollHeight;
    }
  }, [messages, loading]);

  // Initial proactive greeting based on context
  useEffect(() => {
    if (isOpen && context && !hasInitialized) {
      setMessages([]); // Clear previous chat
      setLoading(true);
      
      // Trigger AI to start the conversation based on context PROACTIVELY
      // We pass 'true' as the third argument to indicate this is the init trigger
      sendMessageToGemini([], context, true).then((response) => {
        if (response) {
            setMessages([{ role: 'model', text: response }]);
        } else {
            setMessages([{ role: 'model', text: 'Hallo! Ich bin Matze. Wo drückt der Schuh beim Thema ' + context + '?' }]);
        }
        setLoading(false);
        setHasInitialized(true);
      });

    } else if (!isOpen) {
        setHasInitialized(false);
    }
  }, [isOpen, context]);

  const handleSend = async () => {
    if (!input.trim()) return;

    const newMsg: Message = { role: 'user', text: input };
    const newHistory = [...messages, newMsg];
    
    setMessages(newHistory);
    setInput('');
    setLoading(true);

    // Standard message turn
    const response = await sendMessageToGemini(newHistory, context, false);
    
    setMessages([...newHistory, { role: 'model', text: response || 'Fehler bei der Verbindung.' }]);
    setLoading(false);
  };

  const handleKeyPress = (e: React.KeyboardEvent) => {
    if (e.key === 'Enter') handleSend();
  };

  if (!isOpen) return null;

  return (
    <div className="fixed bottom-5 right-5 z-[3000] w-[90vw] md:w-[400px] h-[500px] bg-white rounded-2xl shadow-2xl border border-gray-200 flex flex-col overflow-hidden font-sans animate-in slide-in-from-bottom-10 fade-in duration-300">
      {/* Header */}
      <div className="bg-primary text-white p-4 flex justify-between items-center shadow-md">
        <div className="flex items-center gap-3">
          <div className="w-10 h-10 rounded-full bg-accent flex items-center justify-center text-primary text-xl shadow-inner border border-white/20">
            <i className="fas fa-hard-hat"></i>
          </div>
          <div>
            <h3 className="font-bold text-sm m-0 leading-tight">Matze</h3>
            <span className="text-[10px] text-gray-300 uppercase tracking-wider flex items-center gap-1">
                <span className="w-1.5 h-1.5 bg-green-400 rounded-full"></span> Online
            </span>
          </div>
        </div>
        <button onClick={onClose} className="text-gray-400 hover:text-white transition-colors">
          <i className="fas fa-times text-xl"></i>
        </button>
      </div>

      {/* Body */}
      <div ref={scrollRef} className="flex-1 p-4 overflow-y-auto bg-[#f8f9fa] space-y-4">
        {messages.map((msg, idx) => (
          <div key={idx} className={`flex ${msg.role === 'user' ? 'justify-end' : 'justify-start'} animate-in fade-in slide-in-from-bottom-2 duration-300`}>
            {msg.role === 'model' && (
                <div className="w-8 h-8 rounded-full bg-primary flex-shrink-0 flex items-center justify-center text-accent text-xs mr-2 mt-1">
                     <i className="fas fa-robot"></i>
                </div>
            )}
            <div 
              className={`max-w-[80%] p-3.5 rounded-2xl text-sm leading-relaxed shadow-sm ${
                msg.role === 'user' 
                  ? 'bg-accent text-white rounded-tr-none' 
                  : 'bg-white text-gray-800 border border-gray-200 rounded-tl-none'
              }`}
            >
              {/* Simple Markdown-like rendering for bullet points */}
              {msg.text.split('\n').map((line, i) => (
                  <p key={i} className={`min-h-[1rem] ${line.trim().startsWith('-') ? 'pl-4' : ''} ${i > 0 ? 'mt-1' : ''}`}>
                      {line}
                  </p>
              ))}
            </div>
          </div>
        ))}
        {loading && (
          <div className="flex justify-start animate-pulse">
            <div className="w-8 h-8 rounded-full bg-primary flex-shrink-0 flex items-center justify-center text-accent text-xs mr-2">
                 <i className="fas fa-robot"></i>
            </div>
            <div className="bg-white p-4 rounded-2xl rounded-tl-none border border-gray-100 shadow-sm flex gap-1.5 items-center h-10">
              <div className="w-1.5 h-1.5 bg-gray-400 rounded-full animate-bounce"></div>
              <div className="w-1.5 h-1.5 bg-gray-400 rounded-full animate-bounce delay-100"></div>
              <div className="w-1.5 h-1.5 bg-gray-400 rounded-full animate-bounce delay-200"></div>
            </div>
          </div>
        )}
      </div>

      {/* Footer */}
      <div className="p-3 bg-white border-t border-gray-100 flex gap-2 shadow-[0_-5px_15px_rgba(0,0,0,0.02)]">
        <input 
          type="text" 
          value={input}
          onChange={(e) => setInput(e.target.value)}
          onKeyDown={handleKeyPress}
          placeholder="Antworte Matze..."
          autoFocus
          className="flex-1 bg-gray-50 text-sm p-3.5 rounded-full border-transparent focus:bg-white focus:border-accent focus:ring-2 focus:ring-accent/20 transition-all outline-none placeholder:text-gray-400"
        />
        <button 
            onClick={handleSend}
            disabled={loading || !input.trim()}
            className="w-11 h-11 rounded-full bg-primary text-accent flex items-center justify-center hover:bg-accent hover:text-white transition-all transform hover:scale-105 active:scale-95 disabled:opacity-50 disabled:cursor-not-allowed shadow-md"
        >
            <i className="fas fa-paper-plane text-sm"></i>
        </button>
      </div>
    </div>
  );
};

export default ChatWidget;